/**************************************************************************************
Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.
***************************************************************************************
$Id: ProfibusDl_Public.h 61384 2016-05-24 10:46:17Z RalfH $

Description:
  definitions of the packet interface to Profibus DL task

Changes:
  Date          Description
  -----------------------------------------------------------------------------------
  2015-01-15    created
**************************************************************************************/

#ifndef __PROFIBUS_DL_PUBLIC_H
#define __PROFIBUS_DL_PUBLIC_H

#include "ProfibusDl_BusParameter.h"

/****************************************************************************************
 * Profibus Dl, constants
 * !!! add additional public constants
 */

#define PROFIBUS_DL_PROCESS_QUEUE_NAME   "PB_DL_QUE"


#define PROFIBUS_DL_MAX_DLPDU_SIZE 246 /* Maximum supported PDU size */

#define PROFIBUS_DL_NIL_SAP 64 /* Signal that the DL command includes Service Access points or not */
#define PROFIBUS_DL_ACCESS_ALL ((TLR_UINT8)~0) /* Access allowed by ALL remote stations */

#define PROFIBUS_DL_MAX_REPLY_UPDATE_SAP 62

#define PROFIBUS_DL_SERVICE_CLASS_MASK  0x01
#define PROFIBUS_DL_SERVICE_CLASS_HIGH  0x01 /* Priority of service */
#define PROFIBUS_DL_SERVICE_CLASS_LOW   0x00
#define PROFIBUS_DL_SERVICE_CLASS_MASK_MSRD  0x02
#define PROFIBUS_DL_SERVICE_CLASS_MSRD  0x02

#define PROFIBUS_DL_SERVICE_CLASS_NOIRQ 0x80


#define PROFIBUS_DL_SERVICE_SDA  0x01 /* Services */
#define PROFIBUS_DL_SERVICE_SDN  0x02
#define PROFIBUS_DL_SERVICE_SRD  0x04
#define PROFIBUS_DL_SERVICE_MSRD 0x08
#define PROFIBUS_DL_SERVICE_CS   0x10

#define PROFIBUS_DL_SERVICE_ROLE_INITIATOR 0x01  /* Role in services */
#define PROFIBUS_DL_SERVICE_ROLE_RESPONDER 0x02
#define PROFIBUS_DL_SERVICE_ROLE_BOTH      0x03

#define PROFIBUS_DL_INDICATION_MODE_ALL       0x00 /* Indication modes */
#define PROFIBUS_DL_INDICATION_MODE_DATA      0x01
#define PROFIBUS_DL_INDICATION_MODE_UNCHANGED 0x02

#define PROFIBUS_DL_REPLY_UPDATE_SINGLE   0x00 /* Reply update mode */
#define PROFIBUS_DL_REPLY_UPDATE_MULTIPLE 0x01

#define PROFIBUS_DL_DATA_SAP_INDEXED 0x80

#define PROFIBUS_DL_UPDATE_STATUS_NO 0x00
#define PROFIBUS_DL_UPDATE_STATUS_LO 0x01
#define PROFIBUS_DL_UPDATE_STATUS_HI 0x02

#define PROFIBUS_DL_STATION_TYPE_NOT_EXISTANT 0x00
#define PROFIBUS_DL_STATION_TYPE_PRESENT  0x01   /* Live List Station is present     */
#define PROFIBUS_DL_STATION_TYPE_PASSIVE  0x01   /* Live List passive Station        */
#define PROFIBUS_DL_STATION_TYPE_READY_NO_TOKEN  0x02   /* Master in ring, no token  */
#define PROFIBUS_DL_STATION_TYPE_ACTIVE   0x03   /* Live List active  Station        */
#define PROFIBUS_DL_STATION_TYPE_MSK      0x03   /* Mask to get the Stationtype bits */

#define PROFIBUS_DL_CLOCKSYNC_TE_CNF      0x01   /* Confirmation of Time Event       */
#define PROFIBUS_DL_CLOCKSYNC_CV_CNF      0x02   /* Confirmation of Clock Value      */
#define PROFIBUS_DL_CLOCKSYNC_CV_MAX_LEN  18     /* Maximum size of clock value data */

#define PROFIBUS_DL_DEVICE_INFO_STRUCT_ID 3074

typedef struct PROFIBUS_XC_CONFIGURATION_Ttag
{
  TLR_HANDLE hxPEC;
  TLR_HANDLE hxMACRpu;
  TLR_HANDLE hxMACTpu;
  TLR_HANDLE hComIrq;
  TLR_HANDLE hPFifo;
  TLR_BOOLEAN32 fRunAsMaster;
  TLR_UINT32 ulMaxComblk;
  TLR_UINT32 ulMaxComDataBlk;
} PROFIBUS_XC_CONFIGURATION_T;


/****************************************************************************************
* Profibus DL, command codes  */

/* Start of the Profibus DL packet area 0x100 - 0x1FF */
#define PROFIBUS_DL_PACKET_COMMAND_START                 0x00000100

/* Summary of the Profibus DL (Datalink Layer Protocol) service commands and primitives */
#define PROFIBUS_DL_CMD_END_PROCESS_REQ                  0x00000100
#define PROFIBUS_DL_CMD_END_PROCESS_CNF                  0x00000101

#define PROFIBUS_DL_CMD_START_DLE_REQ                    0x00000102
#define PROFIBUS_DL_CMD_START_DLE_CNF                    0x00000103

#define PROFIBUS_DL_CMD_STOP_DLE_REQ                     0x00000104
#define PROFIBUS_DL_CMD_STOP_DLE_CNF                     0x00000105

#define PROFIBUS_DL_CMD_DATA_ACK_REQ                     0x00000106
#define PROFIBUS_DL_CMD_DATA_ACK_CNF                     0x00000107

#define PROFIBUS_DL_CMD_DATA_ACK_IND                     0x00000108
#define PROFIBUS_DL_CMD_DATA_ACK_RES                     0x00000109

#define PROFIBUS_DL_CMD_DATA_REQ                         0x0000010A
#define PROFIBUS_DL_CMD_DATA_CNF                         0x0000010B

#define PROFIBUS_DL_CMD_DATA_IND                         0x0000010C
#define PROFIBUS_DL_CMD_DATA_RES                         0x0000010D

#define PROFIBUS_DL_CMD_DATA_REPLY_REQ                   0x0000010E
#define PROFIBUS_DL_CMD_DATA_REPLY_CNF                   0x0000010F

#define PROFIBUS_DL_CMD_DATA_REPLY_IND                   0x00000110
#define PROFIBUS_DL_CMD_DATA_REPLY_RES                   0x00000111

#define PROFIBUS_DL_CMD_DATA_REPLY_UPDATE_REQ            0x00000112
#define PROFIBUS_DL_CMD_DATA_REPLY_UPDATE_CNF            0x00000113

#define PROFIBUS_DL_CMD_MCT_DATA_REPLY_REQ               0x00000114
#define PROFIBUS_DL_CMD_MCT_DATA_REPLY_CNF               0x00000115

#define PROFIBUS_DL_CMD_MCT_DATA_REPLY_IND               0x00000116
#define PROFIBUS_DL_CMD_MCT_DATA_REPLY_RES               0x00000117

#define PROFIBUS_DL_CMD_DXB_DATA_REPLY_IND               0x00000118
#define PROFIBUS_DL_CMD_DXB_DATA_REPLY_RES               0x00000119

#define PROFIBUS_DL_CMD_CS_TIME_EVENT_REQ                0x0000011A
#define PROFIBUS_DL_CMD_CS_TIME_EVENT_CNF                0x0000011B

#define PROFIBUS_DL_CMD_CS_CLOCK_VALUE_REQ               0x0000011C
#define PROFIBUS_DL_CMD_CS_CLOCK_VALUE_CNF               0x0000011D

#define PROFIBUS_DL_CMD_CS_CLOCK_VALUE_IND               0x0000011E
#define PROFIBUS_DL_CMD_CS_CLOCK_VALUE_RES               0x0000011F

#define PROFIBUS_DL_CMD_DLSAP_ACTIVATE_REQ               0x00000120
#define PROFIBUS_DL_CMD_DLSAP_ACTIVATE_CNF               0x00000121

#define PROFIBUS_DL_CMD_DLSAP_ACTIVATE_RESPONDER_REQ     0x00000122
#define PROFIBUS_DL_CMD_DLSAP_ACTIVATE_RESPONDER_CNF     0x00000123

#define PROFIBUS_DL_CMD_DLSAP_ACTIVATE_SUBSCRIBER_REQ    0x00000124
#define PROFIBUS_DL_CMD_DLSAP_ACTIVATE_SUBSCRIBER_CNF    0x00000125

#define PROFIBUS_DL_CMD_SET_VALUE_BUS_PARAMETER_SET_REQ  0x00000126
#define PROFIBUS_DL_CMD_SET_VALUE_BUS_PARAMETER_SET_CNF  0x00000127

#define PROFIBUS_DL_CMD_DLSAP_DEACTIVATE_REQ             0x00000128
#define PROFIBUS_DL_CMD_DLSAP_DEACTIVATE_CNF             0x00000129

#define PROFIBUS_DL_CMD_SET_VALUE_REQ                    0x0000012A
#define PROFIBUS_DL_CMD_SET_VALUE_CNF                    0x0000012B

#define PROFIBUS_DL_CMD_BAUDRATE_TIMER_REQ               0x0000012C
#define PROFIBUS_DL_CMD_BAUDRATE_TIMER_CNF               0x0000012D

#define PROFIBUS_DL_CMD_ABORT_BAUDRATE_DETECTION_REQ     0x0000012E
#define PROFIBUS_DL_CMD_ABORT_BAUDRATE_DETECTION_CNF     0x0000012F

#define PROFIBUS_DL_CMD_GET_LL_REQ                       0x00000130
#define PROFIBUS_DL_CMD_GET_LL_CNF                       0x00000131

#define PROFIBUS_DL_CMD_SEND_FDL_STATUS_REQ              0x00000132
#define PROFIBUS_DL_CMD_SEND_FDL_STATUS_CNF              0x00000133

#define PROFIBUS_DL_CMD_GET_LMS_REQ                      0x00000134
#define PROFIBUS_DL_CMD_GET_LMS_CNF                      0x00000135

#define PROFIBUS_DL_CMD_REGISTER_LMS_REQ                 0x00000136
#define PROFIBUS_DL_CMD_REGISTER_LMS_CNF                 0x00000137

#define PROFIBUS_DL_CMD_LMS_CHANGED_IND                  0x00000138
#define PROFIBUS_DL_CMD_LMS_CHANGED_RES                  0x00000139

#define PROFIBUS_DL_CMD_ABORT_REQ                        0x0000013A
#define PROFIBUS_DL_CMD_ABORT_CNF                        0x0000013B

/*range from 0x160 till 0x17F for standart FDL Task commands (Master) */
#define PROFIBUS_DL_CMD_CS_INIT_REQ                      0x00000160
#define PROFIBUS_DL_CMD_CS_INIT_CNF                      0x00000161

#define PROFIBUS_DL_CMD_CS_TE_REQ                        0x00000162
#define PROFIBUS_DL_CMD_CS_TE_CNF                        0x00000163

#define PROFIBUS_DL_CMD_CS_CV_REQ                        0x00000164
#define PROFIBUS_DL_CMD_CS_CV_CNF                        0x00000165

#define PROFIBUS_DL_CMD_MIN_SLAVE_REQ                    0x00000166
#define PROFIBUS_DL_CMD_MIN_SLAVE_CNF                    0x00000167

/*range from 0x180 till 0x19F reserved for FDL-DLS-Task specific commands (Slave DPV2)*/

#define PROFIBUS_DL_PACKET_COMMAND_END                   0x00000165



/****************************************************************************************
 * Profibus Dl, types
 * !!! add additional public types
 */


/****************************************************************************************
* Profibus Dl, packets */

/* pragma pack */
#ifdef PRAGMA_PACK_ENABLE
#pragma PRAGMA_PACK_1(PROFIBUS_DL_PUBLIC)
#endif

/*
 * Structures of all Request/Confirmation commands the task is able to send and receive
 *
 * Request and Confirmation Packets PROFIBUS_PACKET_DL_xx_REQ/CNF
 * (xx = Command)
 *
 * Use the same order as the commands in TLR_Commands.h
 * PROFIBUS_DL_xx_REQ/CNF (xx = Command)!
 */
/* Request-Packet for the setting the DL parameter */
typedef struct  PROFIBUS_DL_PACKET_SET_BUS_PARAMETER_SET_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_BUS_PARAMETER_SET_T tData;
}
PROFIBUS_DL_PACKET_SET_BUS_PARAMETER_SET_REQ_T;

/* Request-Packet for the stopping the DL */
typedef struct  PROFIBUS_DL_PACKET_STOP_DL_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
}
PROFIBUS_DL_PACKET_STOP_DL_REQ_T;

/* Request-Packet for the starting the DL */
typedef struct  PROFIBUS_DL_PACKET_START_DL_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
}
PROFIBUS_DL_PACKET_START_DL_REQ_T;

typedef __TLR_PACKED_PRE struct  PROFIBUS_DL_DATA_ACK_REQ_Ttag
{
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bDstAddr;   /* Destination address */
  TLR_UINT8 bDstSAPIdx; /* Destination Service Access Point */
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
  TLR_UINT8 abPad[4];   /* Padding needed to bring SDU to a DWORD boundary and to right position */
  TLR_UINT8 abSDU[PROFIBUS_DL_MAX_DLPDU_SIZE]; /* Service Data Unit */
}
__TLR_PACKED_POST PROFIBUS_DL_DATA_ACK_REQ_T;

#define PROFIBUS_DL_DATA_ACK_REQ_SIZE (sizeof(PROFIBUS_DL_DATA_ACK_REQ_T)-PROFIBUS_DL_MAX_DLPDU_SIZE)

/* Request-Packet for acknowledged connectionless data transfer */
typedef struct  PROFIBUS_DL_PACKET_DATA_ACK_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DATA_ACK_REQ_T tData;    /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_DATA_ACK_REQ_T;

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DATA_REQ_Ttag
{
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bDstAddr;   /* Destination address */
  TLR_UINT8 bDstSAPIdx; /* Destination Service Access Point */
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
  TLR_UINT8 abPad[4];   /* Padding needed to bring SDU to a DWORD boundary and to right position */
  TLR_UINT8 abSDU[PROFIBUS_DL_MAX_DLPDU_SIZE]; /* Service Data Unit */
}
__TLR_PACKED_POST PROFIBUS_DL_DATA_REQ_T;

#define PROFIBUS_DL_DATA_REQ_SIZE (sizeof(PROFIBUS_DL_DATA_REQ_T)-PROFIBUS_DL_MAX_DLPDU_SIZE)

/* Request-Packet for unacknowledged connectionless data transfer */
typedef struct  PROFIBUS_DL_PACKET_DATA_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DATA_REQ_T tData;  /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_DATA_REQ_T;

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DATA_REPLY_REQ_Ttag
{
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bDstAddr;   /* Destination address */
  TLR_UINT8 bDstSAPIdx; /* Destination Service Access Point */
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
  TLR_UINT8 abPad[4];   /* Padding needed to bring SDU to a DWORD boundary and to right position */
  TLR_UINT8 abSDU[PROFIBUS_DL_MAX_DLPDU_SIZE]; /* Service Data Unit */
}
__TLR_PACKED_POST PROFIBUS_DL_DATA_REPLY_REQ_T;

#define PROFIBUS_DL_DATA_REPLY_REQ_SIZE (sizeof(PROFIBUS_DL_DATA_REPLY_REQ_T)-PROFIBUS_DL_MAX_DLPDU_SIZE)

/* Request-Packet for two-way connectionless data exchange */
typedef struct  PROFIBUS_DL_PACKET_DATA_REPLY_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DATA_REPLY_REQ_T tData;    /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_DATA_REPLY_REQ_T;

/* Request-Packet for SAP activation */
typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DLSAP_ACTIVATE_REQ_Ttag
{
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point to be activated */
  TLR_UINT8 bAcc;       /* access protection to specify if all stations or individual station may access */
  TLR_UINT8 bParallelServices; /* Number of parallel Services that shall be activated the same time */
  TLR_UINT8 bServiceActivate; /* services to be activated for that SAPIdx SDA,SDN,SRD,MSRD,CS */
  TLR_UINT8 bRoleInService; /* service configuration INITIATOR, RESPONDER,BOTH */
  TLR_UINT8 bMaxDLSDULenReqLow; /* DLSDU length list */
  TLR_UINT8 bMaxDLSDULenReqHigh;
  TLR_UINT8 bMaxDLSDULenIndCnfLow;
  TLR_UINT8 bMaxDLSDULenIndCnfHigh;
}
__TLR_PACKED_POST PROFIBUS_DL_DLSAP_ACTIVATE_REQ_T;

typedef struct  PROFIBUS_DL_PACKET_DLSAP_ACTIVATE_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DLSAP_ACTIVATE_REQ_T tData;
}
PROFIBUS_DL_PACKET_DLSAP_ACTIVATE_REQ_T;


/* Request-Packet for SAP deactivation */
typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DLSAP_DEACTIVATE_REQ_Ttag
{
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point to be activated */
}
__TLR_PACKED_POST PROFIBUS_DL_DLSAP_DEACTIVATE_REQ_T;

typedef struct  PROFIBUS_DL_PACKET_DLSAP_DEACTIVATE_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DLSAP_DEACTIVATE_REQ_T tData;
}
PROFIBUS_DL_PACKET_DLSAP_DEACTIVATE_REQ_T;


/* Request-Packet for SAP responder activation */
typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DLSAP_ACTIVATE_RESPONDER_REQ_Ttag
{
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point to be activated */
  TLR_UINT8 bAcc;       /* access protection to specify if all stations or individual station may access */
  TLR_UINT8 bParallelServices; /* Number of parallel Services that shall be activated the same time */
  TLR_UINT8 bMaxDLSDULenReqLow; /* DLSDU length list */
  TLR_UINT8 bMaxDLSDULenReqHigh;
  TLR_UINT8 bMaxDLSDULenIndLow;
  TLR_UINT8 bMaxDLSDULenIndHigh;
  TLR_UINT8 bIndicationMode;
  TLR_BOOLEAN8 fPublisherEnabled;
}
__TLR_PACKED_POST PROFIBUS_DL_DLSAP_ACTIVATE_RESPONDER_REQ_T;

typedef struct  PROFIBUS_DL_PACKET_DLSAP_ACTIVATE_RESPONDER_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DLSAP_ACTIVATE_RESPONDER_REQ_T tData;
}
PROFIBUS_DL_PACKET_DLSAP_ACTIVATE_RESPONDER_REQ_T;


typedef __TLR_PACKED_PRE struct PROFIBUS_DL_REPLY_UPDATE_REQ_Ttag
{
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bTransmitStrategy;   /* One shot or multiple transmissions */
  TLR_UINT8 abPad[1];   /* Padding needed to bring SDU to a DWORD boundary and to right position */
  TLR_UINT8 abSDU[PROFIBUS_DL_MAX_DLPDU_SIZE]; /* Service Data Unit */
}
__TLR_PACKED_POST PROFIBUS_DL_REPLY_UPDATE_REQ_T;

#define PROFIBUS_DL_REPLY_UPDATE_REQ_SIZE (sizeof(PROFIBUS_DL_REPLY_UPDATE_REQ_T)-PROFIBUS_DL_MAX_DLPDU_SIZE)

/* Request-Packet for updating the reply update buffer of a two-way connectionless data exchange */
typedef struct  PROFIBUS_DL_PACKET_REPLY_UPDATE_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_REPLY_UPDATE_REQ_T tData;    /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_REPLY_UPDATE_REQ_T;

#define PROFIBUS_DL_SET_VALUE_MINTDSR (0)
#define PROFIBUS_DL_SET_VALUE_DLADDR (1)

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_SET_VALUE_REQ_Ttag
{
  TLR_UINT8 bVlu; /* Value to be set */
  __TLR_PACKED_PRE union {
    TLR_UINT16 usMin_TSDR;
    TLR_UINT8 bDl_Add;
  }__TLR_PACKED_POST un;
}
__TLR_PACKED_POST PROFIBUS_DL_SET_VALUE_REQ_T;

/* Request-Packet for setting a DL value */
typedef struct  PROFIBUS_DL_PACKET_SET_VALUE_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_SET_VALUE_REQ_T tData;
}
PROFIBUS_DL_PACKET_SET_VALUE_REQ_T;

/* Request-Packet to get list of active station */
typedef struct  PROFIBUS_DL_PACKET_GET_LL_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
}
PROFIBUS_DL_PACKET_GET_LL_REQ_T;
#define PROFIBUS_DL_GET_LL_REQ_SIZE (0)

/* Request-Packet to get fdl status */
typedef __TLR_PACKED_PRE struct PROFIBUS_DL_SEND_FDL_STATUS_REQ_Ttag
{
  TLR_UINT8 bDstAddr; /* Value to be set */
}
__TLR_PACKED_POST PROFIBUS_DL_SEND_FDL_STATUS_REQ_T;

typedef struct  PROFIBUS_DL_PACKET_SEND_FDL_STATUS_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_SEND_FDL_STATUS_REQ_T tData;
}
PROFIBUS_DL_PACKET_SEND_FDL_STATUS_REQ_T;

#define PROFIBUS_DL_SEND_FDL_STATUS_REQ_SIZE (sizeof(PROFIBUS_DL_SEND_FDL_STATUS_REQ_T))

/* Get List of active stations */
typedef struct  PROFIBUS_DL_PACKET_GET_LMS_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
}
PROFIBUS_DL_PACKET_GET_LMS_REQ_T;

#define PROFIBUS_DL_GET_LMS_REQ_SIZE (0)

/* Register a application to receive LMS change indications*/
typedef __TLR_PACKED_PRE struct PROFIBUS_DL_REGISTER_LMS_REQ_Ttag
{
  TLR_BOOLEAN8 bEnable; /* Value to be set */
}
__TLR_PACKED_POST PROFIBUS_DL_REGISTER_LMS_REQ_T;

typedef struct  PROFIBUS_DL_PACKET_REGISTER_LMS_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_REGISTER_LMS_REQ_T tData;
}
PROFIBUS_DL_PACKET_REGISTER_LMS_REQ_T;

#define PROFIBUS_DL_REGISTER_LMS_REQ_SIZE (sizeof(PROFIBUS_DL_REGISTER_LMS_REQ_T))

/*
 * confirmation packets
 * structures of all confirmation commands the task is able to receive
 */

/* Confirmation-Packet for the setting the DL parameter */
typedef struct  PROFIBUS_DL_PACKET_SET_BUS_PARAMETER_SET_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_BUS_PARAMETER_SET_T tData;
}
PROFIBUS_DL_PACKET_SET_BUS_PARAMETER_SET_CNF_T;

/* Confirmation-Packet for the stopping the DL */
typedef struct  PROFIBUS_DL_PACKET_STOP_DL_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
}
PROFIBUS_DL_PACKET_STOP_DL_CNF_T;

/* Confirmation-Packet for the starting the DL */
typedef struct  PROFIBUS_DL_PACKET_START_DL_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
}
PROFIBUS_DL_PACKET_START_DL_CNF_T;

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DATA_ACK_CNF_Ttag
{
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bDstAddr;   /* Destination address */
  TLR_UINT8 bDstSAPIdx; /* Destination Service Access Point */
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
}
__TLR_PACKED_POST PROFIBUS_DL_DATA_ACK_CNF_T;

/* Confirmation-Packet for acknowledged connectionless data transfer */
typedef struct  PROFIBUS_DL_PACKET_DATA_ACK_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DATA_ACK_CNF_T tData;    /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_DATA_ACK_CNF_T;


typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DATA_CNF_Ttag
{
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bDstAddr;   /* Destination address */
  TLR_UINT8 bDstSAPIdx; /* Destination Service Access Point */
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
}
__TLR_PACKED_POST PROFIBUS_DL_DATA_CNF_T;

/* Confirmation-Packet for unacknowledged connectionless data transfer */
typedef struct  PROFIBUS_DL_PACKET_DATA_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DATA_CNF_T tData;    /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_DATA_CNF_T;

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DATA_REPLY_CNF_Ttag
{
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bDstAddr;   /* Destination address */
  TLR_UINT8 bDstSAPIdx; /* Destination Service Access Point */
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
  TLR_UINT8 abPad[4];   /* Padding needed to bring SDU to a DWORD boundary and to right position */
  TLR_UINT8 abSDU[PROFIBUS_DL_MAX_DLPDU_SIZE]; /* Service Data Unit */
}
__TLR_PACKED_POST PROFIBUS_DL_DATA_REPLY_CNF_T;

#define PROFIBUS_DL_DATA_REPLY_CNF_SIZE (sizeof(PROFIBUS_DL_DATA_REPLY_CNF_T)-PROFIBUS_DL_MAX_DLPDU_SIZE)

/* Confirmation-Packet for two-way connectionless data exchange */
typedef struct  PROFIBUS_DL_PACKET_DATA_REPLY_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DATA_REPLY_CNF_T tData;    /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_DATA_REPLY_CNF_T;

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DLSAP_ACTIVATE_CNF_Ttag
{
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point to be activated */
  TLR_UINT8 bAcc;       /* access protection to specify if all stations or individual station may access */
}
__TLR_PACKED_POST PROFIBUS_DL_DLSAP_ACTIVATE_CNF_T;

/* Confirmation-Packet for SAP activation */
typedef struct  PROFIBUS_DL_PACKET_DLSAP_ACTIVATE_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DLSAP_ACTIVATE_CNF_T tData;    /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_DLSAP_ACTIVATE_CNF_T;

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DLSAP_ACTIVATE_RESPONDER_CNF_Ttag
{
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point to be activated */
  TLR_UINT8 bAcc;       /* access protection to specify if all stations or individual station may access */
}
__TLR_PACKED_POST PROFIBUS_DL_DLSAP_ACTIVATE_RESPONDER_CNF_T;

/* Request-Packet for SAP responder activation */
typedef struct  PROFIBUS_DL_PACKET_DLSAP_ACTIVATE_RESPONDER_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DLSAP_ACTIVATE_RESPONDER_CNF_T tData;    /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_DLSAP_ACTIVATE_RESPONDER_CNF_T;

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DLSAP_DEACTIVATE_CNF_Ttag
{
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point to be activated */
}
__TLR_PACKED_POST PROFIBUS_DL_DLSAP_DEACTIVATE_CNF_T;

/* Confirmation-Packet for SAP deactivation */
typedef struct  PROFIBUS_DL_PACKET_DLSAP_DEACTIVATE_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DLSAP_DEACTIVATE_CNF_T tData;    /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_DLSAP_DEACTIVATE_CNF_T;

/* Confirmation-Packet for setting a DL value */
typedef struct  PROFIBUS_DL_PACKET_SET_VALUE_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
}
PROFIBUS_DL_PACKET_SET_VALUE_CNF_T;

/* Confirmation-Packet for get list of active station */
typedef struct  PROFIBUS_DL_GET_LL_CNF_Ttag
{
  TLR_UINT8 abLL[128];
}
PROFIBUS_DL_GET_LL_CNF_T;

typedef struct  PROFIBUS_DL_PACKET_GET_LL_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_GET_LL_CNF_T tData;     /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_GET_LL_CNF_T;

/* Confirmation-Packet for get fdl status*/
typedef struct __TLR_PACKED_PRE PROFIBUS_DL_SEND_FDL_STATUS_CNF_Ttag
{
  TLR_UINT8 bStatus;
}
__TLR_PACKED_POST PROFIBUS_DL_SEND_FDL_STATUS_CNF_T;

typedef struct  PROFIBUS_DL_PACKET_SEND_FDL_STATUS_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_SEND_FDL_STATUS_CNF_T tData;     /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_SEND_FDL_STATUS_CNF_T;
#define PROFIBUS_DL_SEND_FDL_STATUS_CNF_SIZE (sizeof(PROFIBUS_DL_SEND_FDL_STATUS_CNF_T))


/* Confirmation Packet for get LMS */
typedef struct __TLR_PACKED_PRE PROFIBUS_DL_GET_LMS_CNF_Ttag
{
  TLR_UINT8 abLms[128];
}
__TLR_PACKED_POST PROFIBUS_DL_GET_LMS_CNF_T;

typedef struct  PROFIBUS_DL_PACKET_GET_LMS_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_GET_LMS_CNF_T tData;     /* Packet Data Unit */
} PROFIBUS_DL_PACKET_GET_LMS_CNF_T;

#define PROFIBUS_DL_GET_LMS_CNF_SIZE (sizeof(PROFIBUS_DL_GET_LMS_CNF_T))

/* Register a application to receive LMS change indications*/
typedef struct  PROFIBUS_DL_PACKET_REGISTER_LMS_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
} PROFIBUS_DL_PACKET_REGISTER_LMS_CNF_T;

#define PROFIBUS_DL_REGISTER_LMS_CNF_SIZE (0)

/*
 * indication packets
 * structures of all indication commands the task is able to sent
 */
typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DATA_IND_Ttag
{
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bDstAddr;   /* Destination address */
  TLR_UINT8 bDstSAPIdx; /* Destination Service Access Point */
  TLR_UINT8 bSrcAddr;   /* Source address */
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
  TLR_UINT8 abPad[3];   /* Padding needed to bring SDU to a DWORD boundary and to right position */
  TLR_UINT8 abSDU[PROFIBUS_DL_MAX_DLPDU_SIZE]; /* Service Data Unit */
}
__TLR_PACKED_POST PROFIBUS_DL_DATA_IND_T;

#define PROFIBUS_DL_DATA_IND_SIZE (sizeof(PROFIBUS_DL_DATA_IND_T)-PROFIBUS_DL_MAX_DLPDU_SIZE)

/* Indication Packet for unacknowledged connectionless data transfer */
typedef struct  PROFIBUS_DL_PACKET_DATA_IND_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DATA_IND_T tData;
}
PROFIBUS_DL_PACKET_DATA_IND_T;

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DATA_ACK_IND_Ttag
{
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bDstAddr;   /* Destination address */
  TLR_UINT8 bDstSAPIdx; /* Destination Service Access Point */
  TLR_UINT8 bSrcAddr;   /* Source address */
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
  TLR_UINT8 abPad[3];   /* Padding needed to bring SDU to a DWORD boundary and to right position */
  TLR_UINT8 abSDU[PROFIBUS_DL_MAX_DLPDU_SIZE]; /* Service Data Unit */
}
__TLR_PACKED_POST PROFIBUS_DL_DATA_ACK_IND_T;

#define PROFIBUS_DL_DATA_ACK_IND_SIZE (sizeof(PROFIBUS_DL_DATA_ACK_IND_T)-PROFIBUS_DL_MAX_DLPDU_SIZE)

/* Indication Packet for acknowledged connectionless data transfer */
typedef struct  PROFIBUS_DL_PACKET_DATA_ACK_IND_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DATA_ACK_IND_T tData;
}
PROFIBUS_DL_PACKET_DATA_ACK_IND_T;

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DATA_REPLY_IND_Ttag
{
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bDstAddr;   /* Destination address */
  TLR_UINT8 bDstSAPIdx; /* Destination Service Access Point */
  TLR_UINT8 bSrcAddr;   /* Source address */
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
  TLR_UINT8 bUpdateStatus;   /* Indicates whether or not a response data has been passed */
  TLR_UINT8 abPad[2];   /* Padding needed to bring SDU to a DWORD boundary and to right position */
  TLR_UINT8 abSDU[PROFIBUS_DL_MAX_DLPDU_SIZE]; /* Service Data Unit */
}
__TLR_PACKED_POST PROFIBUS_DL_DATA_REPLY_IND_T;

#define PROFIBUS_DL_DATA_REPLY_IND_SIZE (sizeof(PROFIBUS_DL_DATA_REPLY_IND_T)-PROFIBUS_DL_MAX_DLPDU_SIZE)

/* Indication Packet for acknowledged connectionless data transfer */
typedef struct  PROFIBUS_DL_PACKET_DATA_REPLY_IND_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_DATA_REPLY_IND_T tData;
}
PROFIBUS_DL_PACKET_DATA_REPLY_IND_T;

/* Indication for changed LMS */
typedef __TLR_PACKED_PRE struct PROFIBUS_DL_LMS_CHANGED_IND_Ttag
{
  TLR_UINT8 abLms[128];
}
__TLR_PACKED_POST PROFIBUS_DL_LMS_CHANGED_IND_T;

#define PROFIBUS_DL_LMS_CHANGED_IND_SIZE (sizeof(PROFIBUS_DL_LMS_CHANGED_IND_T))

/* Indication Packet for acknowledged connectionless data transfer */
typedef struct  PROFIBUS_DL_PACKET_LMS_CHANGED_IND_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_LMS_CHANGED_IND_T tData;
}
PROFIBUS_DL_PACKET_LMS_CHANGED_IND_T;


typedef __TLR_PACKED_PRE struct PROFIBUS_DL_REPLY_UPDATE_CNF_Ttag
{
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
  TLR_UINT8 bSrvCls;    /* Service Class */
}
__TLR_PACKED_POST PROFIBUS_DL_REPLY_UPDATE_CNF_T;

/* Confirmation-Packet for after updating the reply update buffer of a two-way connectionless data exchange */
typedef struct  PROFIBUS_DL_PACKET_REPLY_UPDATE_CNF_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_REPLY_UPDATE_CNF_T tData;     /* Packet Data Unit */
}
PROFIBUS_DL_PACKET_REPLY_UPDATE_CNF_T;


/****************************************************************************************
* Profibus DL, API  */

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_API_DATA_IND_Ttag
{
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bDstAddr;   /* Destination address */
  TLR_UINT8 bDstSAPIdx; /* Destination Service Access Point */
  TLR_UINT8 bSrcAddr;   /* Source address */
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
  TLR_UINT8 bUpdateStatus; /* Indicates whether or not a response data has been passed */
}__TLR_PACKED_POST PROFIBUS_DL_API_DATA_IND_T;

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_API_REPLY_UPDATE_REQ_Ttag
{
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bTransmitStrategy;   /* One shot or multiple transmissions */
  TLR_UINT8 bSDULen;    /*Length of SDU */
  TLR_UINT8 abSDU[PROFIBUS_DL_MAX_DLPDU_SIZE]; /* Service Data Unit */
}__TLR_PACKED_POST PROFIBUS_DL_API_REPLY_UPDATE_REQ_T;

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_API_DATA_REQ_Ttag
{
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bDstAddr;   /* Destination address */
  TLR_UINT8 bDstSAPIdx; /* Destination Service Access Point */
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
  TLR_UINT8 abPad[3];   /* Padding needed to bring SDU to a DWORD boundary and to right position */
  TLR_UINT8 bSDULen;    /*Length of SDU */
  TLR_UINT8 abSDU[PROFIBUS_DL_MAX_DLPDU_SIZE]; /* Service Data Unit */
}__TLR_PACKED_POST PROFIBUS_DL_API_DATA_REQ_T;

typedef __TLR_PACKED_PRE struct PROFIBUS_DL_API_DATA_CNF_Ttag
{
  TLR_UINT8 bSrvCls;    /* Service Class */
  TLR_UINT8 bDstAddr;   /* Destination address */
  TLR_UINT8 bDstSAPIdx; /* Destination Service Access Point */
  TLR_UINT8 bSrcSAPIdx; /* Source Service Access Point */
}
__TLR_PACKED_POST PROFIBUS_DL_API_DATA_CNF_T;

typedef struct PROFIBUS_DL_CS_INIT_REQ_Ttag
{
  TLR_UINT32 ulClockSyncCallback;
  TLR_UINT32 ulResouces;
}PROFIBUS_DL_CS_INIT_REQ_T;

typedef struct PROFIBUS_DL_PACKET_CS_INIT_REQ_Ttag
{
  TLR_PACKET_HEADER_T tHead;
  PROFIBUS_DL_CS_INIT_REQ_T tData;
}PROFIBUS_DL_PACKET_CS_INIT_REQ_T;


typedef __TLR_PACKED_PRE struct PROFIBUS_DL_DIAGNOSTIC_Ttag
{
    TLR_UINT32 ulRxValidFrameCnt;
    TLR_UINT32 ulRxValidTokenCnt;
    TLR_UINT32 ulRxErrCnt;
    TLR_UINT32 ulRxRrrStartDelimiterCnt;
    TLR_UINT32 ulTxValidFrameCnt;
    TLR_UINT32 ulTxErrCnt;
    TLR_UINT32 ulDleStarted;
    TLR_UINT32 ulValidBusPrm;
}__TLR_PACKED_POST PROFIBUS_DL_DIAGNOSTIC_T;

typedef struct PROFIBUS_DL_API_CS_REQ_Ttag
{
  TLR_UINT8 bSDULen;    /*Length of SDU */
  TLR_UINT8 abSDU[PROFIBUS_DL_CLOCKSYNC_CV_MAX_LEN]; /* Service Data Unit */
}PROFIBUS_DL_API_CS_REQ_T;

typedef void(*PROFIBUS_DL_DATA_IND_CALLBACK)(TLR_UINT32 ulCmd, PROFIBUS_DL_API_DATA_IND_T *ptDataInd, TLR_UINT32 ulSDULen, TLR_UINT8 *pbSDU, void* pvUser);
typedef void(*PROFIBUS_DL_DATA_CNF_CALLBACK)(TLR_UINT32 ulCmd, TLR_UINT32 ulSta, PROFIBUS_DL_API_DATA_CNF_T *ptDataInd, TLR_UINT32 ulSDULen, TLR_UINT8 *pbSDU, void* pvUser);
typedef void(*PROFIBUS_DL_FDLSTATUS_CNF_CALLBACK)(TLR_UINT32 ulCmd, TLR_UINT32 ulSta, TLR_UINT32 ulRemAdr, TLR_UINT32 ulStatus, void* pvUser);
typedef void(*PROFIBUS_DL_CS_CNF_CALLBACK)(TLR_UINT32 ulCmd, TLR_UINT32 ulSta, void * pvUser);
typedef void(*PROFIBUS_DL_EVENT_CALLBACK)(TLR_UINT32 ulEvent, TLR_UINT32 ulStatus, void* pvUser);

TLR_RESULT ProfibusDl_XCInit (TLR_UINT32 ulXc, PROFIBUS_XC_CONFIGURATION_T* ptCfg);
TLR_RESULT ProfibusDl_XCDeInit (TLR_UINT32 ulXc);

TLR_RESULT ProfibusDl_Open  (TLR_HANDLE *hDlHandle, TLR_UINT32 ulXc);
TLR_RESULT ProfibusDl_Close (TLR_HANDLE hDlHandle);

TLR_RESULT ProfibusDl_GetDiagnostic(TLR_HANDLE hDlHandle, PROFIBUS_DL_DIAGNOSTIC_T* ptDiagnostic);

TLR_RESULT ProfibusDl_StartDLE (TLR_HANDLE hDlHandle);
TLR_RESULT ProfibusDl_StopDLE  (TLR_HANDLE hDlHandle);

TLR_RESULT ProfibusDl_SapActivate (TLR_HANDLE hDlHandle, PROFIBUS_DL_DLSAP_ACTIVATE_REQ_T *ptSAPCfg, PROFIBUS_DL_DATA_IND_CALLBACK pfnCallback, void* pvUser);
TLR_RESULT ProfibusDl_SapDeactivate (TLR_HANDLE hDlHandle, TLR_UINT32 ulSrcSAPIdx);
TLR_RESULT ProfibusDl_RSapActivate (TLR_HANDLE hDlHandle, PROFIBUS_DL_DLSAP_ACTIVATE_RESPONDER_REQ_T *ptRSAPCfg, PROFIBUS_DL_DATA_IND_CALLBACK pfnCallback, void* pvUser);

TLR_RESULT ProfibusDl_DataReplyUpdate (TLR_HANDLE hDlHandle, PROFIBUS_DL_API_REPLY_UPDATE_REQ_T* ptReplyUpd );

TLR_RESULT ProfibusDl_SetValue        (TLR_HANDLE hDlHandle, PROFIBUS_DL_SET_VALUE_REQ_T* ptSetValue);
TLR_RESULT ProfibusDl_SetBusparameter (TLR_HANDLE hDlHandle, PROFIBUS_DL_BUS_PARAMETER_SET_T* ptBusPrm, TLR_UINT32 ulAutoBaud, TLR_UINT32 ulListenMode);

TLR_RESULT ProfibusDl_DataReplyReq (TLR_HANDLE hDlHandle, PROFIBUS_DL_API_DATA_REQ_T *ptDataReq, PROFIBUS_DL_DATA_CNF_CALLBACK pfnCallback, void* pvUser);
TLR_RESULT ProfibusDl_DataAckReq   (TLR_HANDLE hDlHandle, PROFIBUS_DL_API_DATA_REQ_T *ptDataReq, PROFIBUS_DL_DATA_CNF_CALLBACK pfnCallback, void* pvUser);
TLR_RESULT ProfibusDl_DataReq      (TLR_HANDLE hDlHandle, PROFIBUS_DL_API_DATA_REQ_T *ptDataReq, PROFIBUS_DL_DATA_CNF_CALLBACK pfnCallback, void* pvUser);

TLR_RESULT ProfibusDl_GetLMS  							(TLR_HANDLE hDlHandle,
                               							 unsigned char *pabLms,
                               							 unsigned long ulLmsSize);

TLR_RESULT  ProfibusDl_SendFdlStatusReq     (TLR_HANDLE hDlHandle,
                                             TLR_UINT32 ulDstAddr,
                                             PROFIBUS_DL_FDLSTATUS_CNF_CALLBACK pfnCallback,
                                             void* pvUser);

TLR_RESULT ProfibusDl_RegisterLMS           (TLR_HANDLE hDlHandle);

TLR_RESULT ProfibusDl_RegisterEventCallback (TLR_HANDLE hDlHandle,
                                             TLR_UINT32 ulSetEvent,
                                             TLR_UINT32 ulClrEvent,
                                             PROFIBUS_DL_EVENT_CALLBACK pfnCallback,
                                             void* pvUser);

TLR_RESULT ProfibusDl_InstallCsCallback(TLR_HANDLE hDlHandle, PROFIBUS_DL_CS_CNF_CALLBACK pfnCallback, void* pvUser);
TLR_RESULT ProfibusDl_ClockSyncTimeEvent(TLR_HANDLE hDlHandle);
TLR_RESULT ProfibusDl_ClockSyncClockValue(TLR_HANDLE hDlHandle, TLR_UINT8 * ptData, TLR_UINT32 ulTimeSec, TLR_UINT32 ulTimeNSec, TLR_UINT32 ulSystimeRefSec, TLR_UINT32 ulSystimeRefNSec);
TLR_RESULT ProfibusDl_TimeEventReq(TLR_HANDLE hDlHandle, PROFIBUS_DL_API_CS_REQ_T *ptDataReq, PROFIBUS_DL_CS_CNF_CALLBACK pfnCallback, void* pvUser);

void ProfibusDl_SYSTIME_GetSystime( unsigned long* pulSystime_s, unsigned long* pulSystime_ns );

TLR_RESULT ProfibusDL_SetHALInterface(TLR_UINT32 ulXc, void *ptHalIntf);

TLR_RESULT ProfibusDl_CheckBusParameter(PROFIBUS_DL_BUS_PARAMETER_SET_T *ptBusPrm);

/* not implemented yet */
TLR_RESULT ProfibusDl_MCTDataReplyReq (TLR_HANDLE hDlHandle);
TLR_RESULT ProfibusDl_CSTimeEventReq  (TLR_HANDLE hDlHandle);
TLR_RESULT ProfibusDl_CDClockValueReq (TLR_HANDLE hDlHandle);
TLR_RESULT ProfibusDl_SapActivateSubscriber (TLR_HANDLE hDlHandle);

/* pragma unpack */
#ifdef PRAGMA_PACK_ENABLE
#pragma PRAGMA_UNPACK_1(PROFIBUS_DL_PUBLIC)
#endif



/****************************************************************************************
 * Profibus Dl, prototypes
 * !!! add additional public functions (prototypes)
 */


/***************************************************************************************/
#endif /* #ifndef __PROFIBUS_DL_PUBLIC_H */
